﻿using System.Collections.Generic;
using System.Data;
using System.Data.SqlClient;
using System.Linq;

namespace VIRP.EFR.DAL
{
    public class ImagingDataExportDB : DBUtils
    {
        private const string ColumnNameBase = "{0}_{1}";

        private const string SpImageDataExport = "EFR.usp_ImageDataExport";
        private const string SpImageDataExportReferenceData = "EFR.usp_ImageDataExportReferenceData";

        private enum ImageReportDataSetTable
        {
            Summary = 0,
            Details = 1,
            BodyPartsImaged = 2,
            ImagingReasons = 3
        }

        private enum ImageReportReferenceDataSetTable
        {
            BodyPartsImaged = 0,
            ImagingReasons = 1
        }

        private static IReadOnlyList<string> IgnoreColumns = new List<string>
        {
            "PATIENT_ID",
            "PATIENT_ICN",
            "PATIENT_IMAGE_ID"
        };

        private IList<string> StdImageBodyPartsRef;
        private IList<string> StdImageBodyParts
        {
            get
            {
                return StdImageBodyPartsRef;
            }
        }

        private IList<string> StdImageReasonsRef;
        private IList<string> StdImageReasons
        {
            get
            {
                return StdImageReasonsRef;
            }
        }

        internal DataTable GetImagingDataExport(string currentUser, int currentRegistryId)
        {
            LoadReferenceData(currentUser, currentRegistryId);

            var imageDataSet = this.GetImageDataSet(currentUser, currentRegistryId);

            return this.ConvertDataSetToResultTable(imageDataSet);
        }

        private void LoadReferenceData(string currentUser, int currentRegistryId)
        {
            if (StdImageBodyPartsRef == null || StdImageReasonsRef == null)
            {
                StdImageBodyPartsRef = new List<string>();
                StdImageReasonsRef = new List<string>();

                GetReferenceData(currentUser, currentRegistryId);
            }
        }

        private void GetReferenceData(string currentUser, int currentRegistryId)
        {
            using (var ds = GetDataSet(SpImageDataExportReferenceData, currentUser, currentRegistryId))
            {
                foreach (DataRow row in ds.Tables[(int)ImageReportReferenceDataSetTable.BodyPartsImaged].Rows)
                    StdImageBodyPartsRef.Add(row.Field<string>("DESCRIPTION_TEXT"));

                foreach (DataRow row in ds.Tables[(int)ImageReportReferenceDataSetTable.ImagingReasons].Rows)
                    StdImageReasonsRef.Add(row.Field<string>("DESCRIPTION_TEXT"));
            }
        }

        private DataSet GetImageDataSet(string currentUser, int currentRegistryId)
        {
            return GetDataSet(SpImageDataExport, currentUser, currentRegistryId);
        }

        private DataSet GetDataSet(string storedProcedureName, string currentUser, int currentRegistryId)
        {
            DataSet result = new DataSet();

            using (SqlConnection connection = new SqlConnection(this.SqlConnectionString))
            using (SqlCommand cmd = new SqlCommand(storedProcedureName, connection))
            {
                cmd.CommandType = CommandType.StoredProcedure;

                cmd.Parameters.AddWithValue("@CurrentUser", currentUser);
                cmd.Parameters.AddWithValue("@CurrentRegistryId", currentRegistryId);

                connection.Open();

                using (var adapter = new SqlDataAdapter(cmd))
                {
                    adapter.Fill(result);
                }

                connection.Close();
            }

            return result;
        }

        private DataTable ConvertDataSetToResultTable(DataSet imageDataSet)
        {
            this.GetImageDetails(imageDataSet.Tables[(int)ImageReportDataSetTable.Summary], imageDataSet);

            imageDataSet.Tables[0].Columns.Remove("PATIENT_ID");

            return imageDataSet.Tables[0];
        }

        private void GetImageDetails(DataTable resultTable, DataSet imageDataSet)
        {
            const string ImageIdBase = "IMG_{0}";

            DataTable imageDetailsTable = imageDataSet.Tables[(int)ImageReportDataSetTable.Details];

            foreach (DataRow resultRow in resultTable.Rows)
            {
                var currentPatientId = resultRow.Field<int>("PATIENT_ID");
                var patientImageCounter = 1;

                foreach (DataRow detailsRow in imageDetailsTable.Rows)
                {
                    if (detailsRow.Field<int>("PATIENT_ID") == currentPatientId)
                    {
                        var patientImageId = detailsRow.Field<int>("PATIENT_IMAGE_ID");
                        var image = string.Format(ImageIdBase, patientImageCounter++);

                        foreach (DataColumn column in imageDetailsTable.Columns)
                        {
                            if (IgnoreColumns.Contains(column.ColumnName)) continue;

                            var currentColumn = string.Format(ColumnNameBase, image, column.ColumnName);
                            if (!resultTable.Columns.Contains(currentColumn))
                                resultTable.Columns.Add(currentColumn, column.DataType);

                            resultRow[currentColumn] = detailsRow[column.ColumnName];
                        }

                        this.GetBodyPartsImaged(resultTable, imageDataSet.Tables[(int)ImageReportDataSetTable.BodyPartsImaged], resultRow, image, patientImageId);
                        this.GetImagingReasons(resultTable, imageDataSet.Tables[(int)ImageReportDataSetTable.ImagingReasons], resultRow, image, patientImageId);
                    }
                }
            }
        }

        private void GetBodyPartsImaged(
            DataTable resultTable,
            DataTable bodyPartsImagedTable,
            DataRow resultRow,
            string image,
            int patientImageId)
        {
            foreach (var part in StdImageBodyParts)
            {
                var columnName = string.Format(ColumnNameBase, image, part);

                if (!resultTable.Columns.Contains(columnName))
                    resultTable.Columns.Add(columnName, typeof(string));

                resultRow[columnName] = "0";
            }

            foreach (DataRow row in bodyPartsImagedTable.Rows)
            {
                if (row.Field<int>("PATIENT_IMAGE_ID") == patientImageId)
                {
                    var columnName = string.Format(ColumnNameBase, image, row.Field<string>("BODY_PART_IMAGED"));
                    resultRow[columnName] = "1";
                }
            }
        }

        private void GetImagingReasons(
            DataTable resultTable,
            DataTable imagingReasonsTable,
            DataRow resultRow,
            string image,
            int patientImageId)
        {
            foreach (var reason in StdImageReasons)
            {
                var columnName = string.Format(ColumnNameBase, image, reason);

                if (!resultTable.Columns.Contains(columnName))
                    resultTable.Columns.Add(columnName, typeof(string));

                resultRow[columnName] = "0";
            }

            foreach (DataRow row in imagingReasonsTable.Rows)
            {
                if (row.Field<int>("PATIENT_IMAGE_ID") == patientImageId)
                {
                    var columnName = string.Format(ColumnNameBase, image, row.Field<string>("IMAGE_REASON"));
                    resultRow[columnName] = "1";
                }
            }
        }
    }
}
